/******************************************************
 *
 * Name:         iaas-sql-server-use-tally-tables.sql
 *     
 * Design Phase:
 *     Author:   John Miner
 *     Date:     01-30-2025
 *     Purpose:  Comparison of row by agonising row (rbar)
 *               and tally table methods to create date dimension.
 * 
 ******************************************************/

--
-- 1 - Create Dimension schema
--

-- Delete existing schema.
IF EXISTS (SELECT * FROM sys.schemas WHERE name = N'DIM')
DROP SCHEMA [DIM]
GO

-- Add new schema.
CREATE SCHEMA [DIM] AUTHORIZATION [dbo]
GO

-- Show the new schema
SELECT * FROM sys.schemas WHERE name = 'DIM';
GO


--
-- 2 - Create date dim table 
--

-- Use the correct database
USE [SSC_ARTILCES];
GO

-- Delete existing table
IF  OBJECT_ID(N'[DIM].[DATE_PROPERTIES]') > 0
    DROP TABLE [DIM].[DATE_PROPERTIES]
GO

-- Create new table
CREATE TABLE [DIM].[DATE_PROPERTIES]
(
  date_key INT IDENTITY(1,1) NOT NULL CONSTRAINT PK_DIM_DATE_PROPERTIES PRIMARY KEY,
  date_date SMALLDATETIME NOT NULL,
  date_string VARCHAR(10) NOT NULL,
  date_month VARCHAR(3) NOT NULL,
  date_day VARCHAR(3) NOT NULL,
  date_int_qtr INT NOT NULL,
  date_int_doy INT NOT NULL,
  date_int_yyyy INT NOT NULL,
  date_int_mm INT NOT NULL,
  date_int_dd INT NOT NULL
);
GO

-- Drop existing index
IF EXISTS (SELECT name FROM sys.indexes WHERE name = N'IX_DIM_DATE_PROPERTIES_STR')
DROP INDEX IX_DIM_DATE_STR ON [DIM].[DATE_PROPERTIES];
GO

-- Create new index
CREATE INDEX IX_DIM_DATE_STR
    ON [DIM].[DATE_PROPERTIES] (date_string); 
GO

-- Show new table
SELECT s.name as schema_nm, T.name as table_nm, T.type_desc
FROM SYS.TABLES as T
JOIN SYS.SCHEMAS as S
ON T.schema_id = S.schema_id
GO


--
-- 3 - Load date dim table (RBAR)
--

-- https://www.red-gate.com/simple-talk/databases/sql-server/t-sql-programming-sql-server/rbar-row-by-agonizing-row/

-- Clear table
TRUNCATE TABLE [DIM].[DATE_PROPERTIES]
GO

-- Turn off counting
SET NOCOUNT ON
GO

-- Create and intialize loop variable
DECLARE @START_DTE DATETIME2(6) = GETDATE();
DECLARE @VAR_DATE SMALLDATETIME = '2020-01-01';

-- Add data to our new table
WHILE (DATEDIFF(D, @VAR_DATE, '2030-01-01') <> 0)
BEGIN

    -- Add row to dimension table
    INSERT INTO [DIM].[DATE_PROPERTIES]
    (
        date_date,
        date_string,
        date_month,
        date_day,
        date_int_qtr,
        date_int_doy,
        date_int_yyyy,
        date_int_mm,
        date_int_dd
    )
    VALUES
	(
	   -- As small date time
	   @VAR_DATE,

	   -- Date as string
	   SUBSTRING(CONVERT(CHAR(10), @VAR_DATE, 120) + REPLICATE(' ', 10), 1, 10), 

	   -- Month as string
	   UPPER(SUBSTRING(CONVERT(CHAR(10), @VAR_DATE, 100) + REPLICATE(' ', 3), 1, 3)),

	   -- Day as string
	   UPPER(SUBSTRING(DATENAME(DW, @VAR_DATE), 1, 3)),

	   -- As quarter of year
	   DATEPART(QQ, @VAR_DATE),

	   -- As day of year
	   DATEPART(DY, @VAR_DATE),

	   -- Year as integer
	   YEAR(@VAR_DATE),

	   -- Month as integer
	   MONTH(@VAR_DATE),

	   -- Day as integer
	   DAY(@VAR_DATE)

    );

    -- Increment the counter
    SELECT @VAR_DATE = DATEADD(D, 1, @VAR_DATE);
END;

-- GRAB TIMINGS 
DECLARE @END_DTE DATETIME2(6) = GETDATE();

SELECT '1 - START DATE/TIME' AS DESCRIPTION_01, CONVERT(CHAR(24), @START_DTE, 113) AS VALUE_01
UNION
SELECT '2 - END DATE/TIME' AS DESCRIPTION_01, CONVERT(CHAR(24), @END_DTE, 113) AS VALUE_01
UNION
SELECT '3 - TOTAL ELAPSED MS' AS DESCRIPTION_01, STR(DATEDIFF(MILLISECOND, @START_DTE, @END_DTE)) AS VALUE_01
GO



--
-- 4 - Load date dim table (TALLY TABLE)
--

-- https://sqlbenjamin.wordpress.com/2018/02/06/sql-tip-the-tally-table/

-- Clear table
TRUNCATE TABLE [DIM].[DATE_PROPERTIES]
GO

-- Turn off counting
SET NOCOUNT ON
GO

-- Grab start date/time
DECLARE @START_DTE DATETIME2(6) = GETDATE();

-- Ten years of dates
WITH CTE_TALLY_TABLE AS
(
	SELECT TOP 100000 
		ROW_NUMBER() OVER(ORDER BY (SELECT NULL)) AS [N]
	FROM 
		dbo.syscolumns tb1
	CROSS JOIN 
		dbo.syscolumns tb2 
),
CTE_TEN_YEARS AS
(
    SELECT
        N AS date_key, 
        CAST(DATEADD(D, N, '2020-01-01') AS DATETIME2(6)) AS date_date
    FROM 
        CTE_TALLY_TABLE
    WHERE 
        N < (365.25 * 10)
),

-- Create properties for a given date
CTE_DATE_TABLE AS
(
    SELECT 
        
        -- row key
        -- t.date_key,

        -- date value
        t.date_date,

        -- Date as string
        SUBSTRING(CONVERT(CHAR(10), t.date_date, 120) + CAST(REPLICATE(' ', 10) AS CHAR(10)), 1, 10) as date_string, 

        -- Month as string
        UPPER(SUBSTRING(CONVERT(CHAR(10), t.date_date, 100) + CAST(REPLICATE(' ', 3) AS CHAR(3)), 1, 3)) as date_month,

        -- Day as string
        CAST(UPPER(SUBSTRING(DATENAME(DW, t.date_date), 1, 3)) AS CHAR(3)) as date_day,

        -- As quarter of year
        DATEPART(QQ, t.date_date) as date_int_qtr,

        -- As day of year
        DATEPART(DY, t.date_date) as date_int_doy,

        -- Year as integer
        YEAR(t.date_date) as date_int_yyyy,

        -- Month as integer
        MONTH(t.date_date) as date_int_mm,

        -- Day as integer
        DAY(t.date_date) as date_int_dd   
    FROM 
        CTE_TEN_YEARS AS t
)

-- Add to dimension table
INSERT INTO [DIM].[DATE_PROPERTIES]
SELECT * FROM CTE_DATE_TABLE;

-- GRAB TIMINGS 
DECLARE @END_DTE DATETIME2(6) = GETDATE();

SELECT '1 - START DATE/TIME' AS DESCRIPTION_01, CONVERT(CHAR(24), @START_DTE, 113) AS VALUE_01
UNION
SELECT '2 - END DATE/TIME' AS DESCRIPTION_01, CONVERT(CHAR(24), @END_DTE, 113) AS VALUE_01
UNION
SELECT '3 - TOTAL ELAPSED MS' AS DESCRIPTION_01, STR(DATEDIFF(MILLISECOND, @START_DTE, @END_DTE)) AS VALUE_01
GO
