/*
Stable Marriages

Run this script to build all tables and procs.

Then run stored proc, eg. exec dbo.spStableMarriage 20
*/

----------------------------------------------------------------------------------------------------------------------------
CREATE PROCEDURE [dbo].[spBuildTables](
@N int
)
AS

/*
Create table BG
Create tables B,G with columns B1,...,B@N and G1,...,G@N
*/

BEGIN

-- Declarations 
DECLARE @I		int
DECLARE @SQL	NVARCHAR(MAX)

-- No counting
SET NOCOUNT ON

-- Build BG
SET @SQL = '
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N''[dbo].[BG]'') AND type in (N''U''))
	DROP TABLE [dbo].[BG]
CREATE TABLE [dbo].[BG](
	[B] [int] NULL,
	[G] [int] NULL
) ON [PRIMARY]'
exec sp_executesql @SQL

-- Build B
SET @SQL = '
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N''[dbo].[B]'') AND type in (N''U''))
	DROP TABLE [dbo].[B]
CREATE TABLE [dbo].[B](
	[B] [int] NULL,
'
SET @I = 0
WHILE @I < @N
	BEGIN
	SET @I = @I + 1
	SET @SQL = @SQL + '[G' + cast(@I AS NVARCHAR(16)) + '] [int] NULL, '
	END 
SET @SQL = left(@SQL,len(@SQL) - 1)
SET @SQL = @SQL + ') ON [PRIMARY]'
exec sp_executesql @SQL

-- Build G
SET @SQL = '
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N''[dbo].[G]'') AND type in (N''U''))
	DROP TABLE [dbo].[G]
CREATE TABLE [dbo].[G](
	[G] [int] NULL,
'
SET @I = 0
WHILE @I < @N
	BEGIN
	SET @I = @I + 1
	SET @SQL = @SQL + '[B' + cast(@I AS NVARCHAR(16)) + '] [int] NULL, '
	END 
SET @SQL = left(@SQL,len(@SQL) - 1)
SET @SQL = @SQL + ') ON [PRIMARY]'
exec sp_executesql @SQL

END

GO
----------------------------------------------------------------------------------------------------------------------------
CREATE PROCEDURE [dbo].[spCheckTables]
AS

/*
Check that B and G have same number boys and girls
Check that B and G have no duplicate, orphan or NULL values
*/

BEGIN

-- Declarations
DECLARE @MSG			VARCHAR(128)
DECLARE @NumBoys		INT
DECLARE @NumGirls		INT
DECLARE @Count			int
DECLARE @ColumnNamesB	NVARCHAR(MAX)
DECLARE @ColumnNamesG	NVARCHAR(MAX)
DECLARE @NumColumns		int
DECLARE @SQL			NVARCHAR(MAX)
DECLARE @ParmDefinition NVARCHAR(MAX)

-- No counting
SET NOCOUNT ON

-- Count the number of boys and girls 
SET @NumBoys = (SELECT count(*) FROM B)
SET @NumGirls = (SELECT count(*) FROM G)

-- If different, return
IF @NumBoys <> @NumGirls 
	BEGIN
	SET @MSG = 'ERROR: Different numbers of boys and girls'
	PRINT @MSG
	RETURN 1
	END

-- Get number of column names for B, G 
SET @NumColumns = @NumBoys

-- Produce SQL snippet containing column names for B, G 
DECLARE @T1 TABLE (ColumnNames NVARCHAR(MAX))
INSERT INTO @T1 exec dbo.spGetColumnNames 'B', @NumColumns 
SET @ColumnNamesB = (SELECT TOP 1 ColumnNames FROM @T1)

DECLARE @T2 TABLE (ColumnNames NVARCHAR(MAX))
INSERT INTO @T2 exec dbo.spGetColumnNames 'G', @NumColumns 
SET @ColumnNamesG = (SELECT TOP 1 ColumnNames FROM @T2)

-- Initialize @Count 
SET @Count = 0

-- If duplicate, missing, orphan or NULL values exist among the columns of some boy, return
SET @SQL = '
SELECT @Count = count(B) FROM
(
SELECT 1 AS B FROM
(
SELECT B, count(*) AS NumCount FROM
(
SELECT DISTINCT d1.B, d2.G FROM
(
SELECT B, G FROM B
UNPIVOT ( G FOR X IN ( ' + @ColumnNamesG + ') ) AS u
) d1
INNER JOIN
G d2
ON 
d1.G = d2.G
) d
GROUP BY B
HAVING
count(*) <> @NumBoys
) d3
) d4'
SET @ParmDefinition = N'@NumBoys int, @Count int OUTPUT'
SET @NumBoys = (SELECT count(*) FROM B)
EXEC dbo.sp_executesql @SQL, @ParmDefinition, @NumBoys = @NumBoys, @Count = @Count OUTPUT;

IF @Count > 0
	BEGIN
	SET @MSG = 'ERROR: Duplicate, orphan or NULL values for some boy'
	PRINT @MSG
	RETURN 2
	END

-- If duplicate, missing, orphan or NULL values exist among the columns of some girl, return
SET @SQL = '
SELECT @Count = count(B) FROM
(
SELECT 1 AS B FROM
(
SELECT G, count(*) AS NumCount FROM
(
SELECT DISTINCT d1.G, d2.B FROM
(
SELECT G, B FROM G
UNPIVOT ( B FOR X IN ( ' + @ColumnNamesB + ' ) ) AS u
) d1
INNER JOIN
B d2
ON 
d1.B = d2.B
) d
GROUP BY G
HAVING
count(*) <> @NumGirls
) d3
) d4'
SET @ParmDefinition = N'@NumGirls int, @Count int OUTPUT'
SET @NumGirls = (SELECT count(*) FROM G)
EXEC dbo.sp_executesql @SQL, @ParmDefinition, @NumGirls = @NumGirls, @Count = @Count OUTPUT;

IF @Count > 0
	BEGIN
	SET @MSG = 'ERROR: Duplicate, orphan or NULL values for some girl'
	PRINT @MSG
	RETURN 3
	END

END

GO
----------------------------------------------------------------------------------------------------------------------------
CREATE PROCEDURE [dbo].[spGetColumnNames](
@Table VARCHAR(128),
@N int
)
AS

/*
Generate SQL snippet of @N column names for @Table = 'B' or 'G'
Eg. B1,B2,B3,B4,B5,B6,B7,B8,B9,B10
*/

BEGIN

-- Declarations 
DECLARE @SNIP	VARCHAR(MAX) 

-- No counting
SET NOCOUNT ON

-- Declarations for sp_executesql
DECLARE @SQL			NVARCHAR(MAX)
DECLARE @ParmDefinition NVARCHAR(MAX)
DECLARE @IN				NVARCHAR(16)
DECLARE @OUT			NVARCHAR(16)
DECLARE @INPUT			NVARCHAR(128)

-- Build script 
SET @SQL = N'
DECLARE @T TABLE (I int)
DECLARE @I int
SET @I = 0
WHILE @I < @IN
	BEGIN
	SET @I = @I + 1
	INSERT INTO @T(I) VALUES(@I)
	END

SET @OUT = 
(
SELECT ''' + @Table + ''' + cast(I as VARCHAR(16)) + '',''
FROM    @T ORDER BY I
FOR     XML PATH('''')
)
SET @OUT = left(@OUT,len(@OUT) - 1)
'

-- Call script with input = @INPUT and output = @SNIP
SET @ParmDefinition = N'@IN int, @OUT VARCHAR(MAX) OUTPUT'
SET @INPUT = cast(@N AS VARCHAR(16))
EXEC dbo.sp_executesql @SQL, @ParmDefinition, @IN = @INPUT, @OUT = @SNIP OUTPUT;

-- Return answer
SELECT @SNIP

END

GO
----------------------------------------------------------------------------------------------------------------------------
CREATE PROCEDURE [dbo].[spPopulateTables]
AS

/*
Populate B, G with manual data entry
*/

BEGIN

-- No counting
SET NOCOUNT ON

-- Populate B
TRUNCATE TABLE B
INSERT INTO B(B,G1,G2,G3,G4,G5) VALUES (1,2,4,5,1,3)
INSERT INTO B(B,G1,G2,G3,G4,G5) VALUES (2,1,5,2,4,3)
INSERT INTO B(B,G1,G2,G3,G4,G5) VALUES (3,2,1,3,5,4)
INSERT INTO B(B,G1,G2,G3,G4,G5) VALUES (4,1,2,3,4,5)
INSERT INTO B(B,G1,G2,G3,G4,G5) VALUES (5,3,4,1,2,5)

-- Populate G
TRUNCATE TABLE G
INSERT INTO G(G,B1,B2,B3,B4,B5) VALUES (1,4,3,5,2,1)
INSERT INTO G(G,B1,B2,B3,B4,B5) VALUES (2,1,2,5,3,4)
INSERT INTO G(G,B1,B2,B3,B4,B5) VALUES (3,2,3,1,5,4)
INSERT INTO G(G,B1,B2,B3,B4,B5) VALUES (4,2,3,4,1,5)
INSERT INTO G(G,B1,B2,B3,B4,B5) VALUES (5,5,2,3,4,1)

END

GO
----------------------------------------------------------------------------------------------------------------------------
CREATE PROCEDURE [dbo].[spRandomPermutation](
@N int
)
AS

/*
Randomly permute numbers 1 to @N
*/

BEGIN

-- Declarations 
DECLARE @PERMUTATION	VARCHAR(MAX)

-- No counting
SET NOCOUNT ON

-- Declarations for sp_executesql
DECLARE @SQL			NVARCHAR(MAX)
DECLARE @ParmDefinition NVARCHAR(MAX)
DECLARE @IN				NVARCHAR(16)
DECLARE @OUT			NVARCHAR(16)
DECLARE @INPUT			NVARCHAR(128)

-- Build script 
SET @SQL = N'
DECLARE @T TABLE (I int)
DECLARE @I int
SET @I = 0
WHILE @I < @IN
	BEGIN
	SET @I = @I + 1
	INSERT INTO @T(I) VALUES(@I)
	END

SET @OUT = 
(
SELECT cast(I as VARCHAR(16)) + '',''
FROM    @T ORDER BY NEWID()
FOR     XML PATH('''')
)
SET @OUT = left(@OUT,len(@OUT) - 1)
'

-- Call script with input = @INPUT and output = @PERMUTATION
SET @ParmDefinition = N'@IN int, @OUT VARCHAR(MAX) OUTPUT'
SET @INPUT = cast(@N AS VARCHAR(16))
EXEC dbo.sp_executesql @SQL, @ParmDefinition, @IN = @INPUT, @OUT = @PERMUTATION OUTPUT;

-- Select permutation
SELECT @PERMUTATION

END

GO
----------------------------------------------------------------------------------------------------------------------------
CREATE PROCEDURE [dbo].[spRandomTable](
@Table VARCHAR(128)
)
AS

/*
Populate rows of table B or G with random permutations 
*/

BEGIN

-- Declarations
DECLARE @I int
DECLARE @T TABLE (I int identity, P NVARCHAR(MAX))
DECLARE @SQL NVARCHAR(MAX)
DECLARE @INSERT NVARCHAR(MAX)
DECLARE @NumColumns int

-- No counting
SET NOCOUNT ON

-- Get number of column names for B, G
SET @NumColumns = (SELECT count(COLUMN_NAME) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = 'B') - 1

-- Populate table @T with random permutations of 1 to @NumColumns
SET @I = 0
WHILE @I < @NumColumns
	BEGIN
	SET @I = @I + 1
	INSERT INTO @T exec dbo.spRandomPermutation @NumColumns
	END

-- Prepare INSERT snippet for table B or G
IF @Table = 'B'
	BEGIN
	SET @INSERT = N'INSERT INTO B VALUES(' 
	TRUNCATE TABLE B
	END
ELSE
	BEGIN
	SET @INSERT = N'INSERT INTO G VALUES(' 
	TRUNCATE TABLE G
	END

-- Declare cursor to yield INSERT queries
DECLARE curTable CURSOR
FOR 
SELECT @INSERT + cast(I AS NVARCHAR(16)) + ',' + P + ')' FROM @T
FOR READ ONLY

-- Open cursor
OPEN curTable 

-- Fetch first row
FETCH NEXT FROM curTable INTO @SQL

-- Execute INSERT statements
While @@FETCH_STATUS = 0 
	BEGIN
	EXEC dbo.sp_executesql @SQL
	FETCH NEXT FROM curTable INTO @SQL
	END

-- Close cursor
CLOSE curTable

-- Deallocate cursor
DEALLOCATE curTable

END

GO
----------------------------------------------------------------------------------------------------------------------------
CREATE PROCEDURE [dbo].[spStableMarriage](
@N int = 0
)
AS

BEGIN

/*
Stable Marriages 

This script will use the preference lists of boys B and girls G
to build stable marriages in BG, where no boy and girl will prefer
each other to their partners.
*/

-- Declarations
DECLARE @B				int
DECLARE @B1				int
DECLARE @G				int
DECLARE @COL			NVARCHAR(16)
DECLARE @COL1			NVARCHAR(16)
DECLARE @COL2			NVARCHAR(16)
DECLARE @DEBUG			bit
DECLARE @MSG			VARCHAR(128)
DECLARE @NumBoys		INT
DECLARE @NumColumns		INT

-- Declarations for sp_executesql
DECLARE @ColumnNamesB	NVARCHAR(MAX)
DECLARE @ColumnNamesG	NVARCHAR(MAX)
DECLARE @SQL			NVARCHAR(MAX)
DECLARE @ParmDefinition NVARCHAR(MAX)
DECLARE @IN				NVARCHAR(16)
DECLARE @OUT			NVARCHAR(16)
DECLARE @INPUT1			NVARCHAR(128)
DECLARE @INPUT2			NVARCHAR(128)
DECLARE @Count			int

-- No counting
SET NOCOUNT ON

-- Set debugging on or off
SET @DEBUG = 1

-- Build empty tables B, G, BG with specified number of boys (and girls)
-- If @N = 0 then fixed data of 5 boys (girls) in spPopulateTables will be used 
-- That way, you can experiment with custom preferences
IF @N = 0 
	exec dbo.spBuildTables 5 -- must equal number of boys (girls) in spPopulateTables
ELSE
	exec dbo.spBuildTables @N

-- Populate empty tables B and G
IF @N = 0 
	-- Manually entered preferences will be used
	-- Run the command: 
	--   exec dbo.spStableMarriage 0 
	-- to use fixed data in spPopulateTables
	exec dbo.spPopulateTables
ELSE
	-- Randomly generated preferences will be used
	-- Run the command: 
	--   exec dbo.spStableMarriage @N
	-- where @N > 0 
	BEGIN
	exec dbo.spRandomTable 'B' -- Randomize B 
	exec dbo.spRandomTable 'G' -- Randomize G
	END

-- Check data integrity if manually entered preferences used
-- Check that B and G have same number of columns
-- Check that B and G have no duplicate, orphan or NULL values
IF @N = 0 
	BEGIN
	DECLARE @RESULT int;  
	EXECUTE @RESULT = dbo.spCheckTables 
	IF @RESULT > 0
		RETURN @RESULT
	END

-- Count number of boys (and girls)
SET @NumBoys = (SELECT count(*) FROM B)

-- Get number of column names for B, G
SET @NumColumns = (SELECT count(COLUMN_NAME) 
FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = 'B') - 1

-- Produce SQL snippets containing column names for B, G 
-- Eg. @ColumnNamesB = B1,B2,B3,B4,B5
-- Eg. @ColumnNamesG = G1,G2,G3,G4,G5
-- These will be used in upcoming queries
DECLARE @T1 TABLE (ColumnNames NVARCHAR(MAX))
INSERT INTO @T1 exec dbo.spGetColumnNames 'B', @NumColumns 
SET @ColumnNamesB = (SELECT TOP 1 ColumnNames FROM @T1)

DECLARE @T2 TABLE (ColumnNames NVARCHAR(MAX))
INSERT INTO @T2 exec dbo.spGetColumnNames 'G', @NumColumns 
SET @ColumnNamesG = (SELECT TOP 1 ColumnNames FROM @T2)

-- Display B before some columns are NULLIFIED below
-- NULL values represent proposals (only last one is successful)
SELECT * FROM B

-- Loop through boys until everyone is engaged
WHILE (SELECT count(*) FROM BG) < @NumBoys
	BEGIN

	-- Find a boy who is not currently engaged  
	-- Note that he may have been engaged earlier but was later dropped for another boy
	SET @B = (SELECT TOP 1 d1.B FROM B d1 LEFT OUTER JOIN BG d2 on d1.B = d2.B WHERE d2.B IS NULL)

	-- Find the column @COL of the first girl on his preference list that's not yet NULL 
	-- This column will be NULLIFIED after he proposes to her, regardless of whether she accepts
	-- That way, he won't propose to her again
	SET @SQL = N'
		SELECT @OUT = 
		(
		SELECT TOP 1 X
		FROM B
		UNPIVOT ( G FOR X IN ( ' + @ColumnNamesG + ' ) ) AS u
		WHERE B = @IN)' 
	SET @ParmDefinition = N'@IN NVARCHAR(16), @OUT NVARCHAR(16) OUTPUT'
	SET @INPUT1 = cast(@B AS NVARCHAR(16))
	EXEC dbo.sp_executesql @SQL, @ParmDefinition, @IN = @INPUT1, @OUT = @COL OUTPUT;

	-- Identify the girl @G in that column
	-- Note that @COL is not required here (but it's used later to NULLIFY its value)
	SET @SQL = 
		N'SELECT @OUT = (SELECT TOP 1 G FROM B UNPIVOT ( G FOR X IN ( ' + @ColumnNamesG + ' ) ) AS u WHERE B = @IN)' 
	SET @ParmDefinition = N'@IN NVARCHAR(16), @OUT int OUTPUT'
	SET @INPUT1 = cast(@B AS NVARCHAR(16))
	EXEC dbo.sp_executesql @SQL, @ParmDefinition, @IN = @INPUT1, @OUT = @G OUTPUT;

	-- Comment 
	SET @SQL = N'SELECT @OUT = count(*) FROM B
				UNPIVOT ( G FOR X IN ( ' + @ColumnNamesG + ' ) ) AS u
				WHERE B = @IN' 
	SET @ParmDefinition = N'@IN NVARCHAR(16), @OUT int OUTPUT'
	SET @INPUT1 = cast(@B AS NVARCHAR(16))
	EXEC dbo.sp_executesql @SQL, @ParmDefinition, @IN = @INPUT1, @OUT = @Count OUTPUT;
	IF @Count = @NumBoys
		SET @MSG =	'Boy ' + cast(@B AS VARCHAR(16)) + ' makes first proposal to Girl ' + 
					cast(@G AS VARCHAR(16))
		ELSE
		SET @MSG =	'Boy ' + cast(@B AS VARCHAR(16)) + ' then proposes to Girl ' +	
					cast(@G AS VARCHAR(16))
	PRINT @MSG

	-- Determine if she's engaged to another boy @B1
	SET @SQL = N'SELECT @OUT = (SELECT TOP 1 B FROM BG WHERE G = @IN)' 
	SET @ParmDefinition = N'@IN NVARCHAR(16), @OUT int OUTPUT'
	SET @INPUT1 = cast(@G AS NVARCHAR(16))
	EXEC dbo.sp_executesql @SQL, @ParmDefinition, @IN = @INPUT1, @OUT = @B1 OUTPUT;

	-- Comment
	IF @B1 IS NULL
		BEGIN
		SET @MSG = 'Girl ' + cast(@G AS VARCHAR(16)) + ' is not currently engaged'
		PRINT @MSG
		END

	IF @B1 IS NULL
		-- If not engaged to another boy @B1, then engage her to new boy @B
		BEGIN	
		INSERT INTO BG(B,G) VALUES (@B,@G)

		-- Comment		
		SET @MSG =	'So Girl ' + cast(@G AS VARCHAR(16)) + ' accepts proposal from Boy ' + 
					cast(@B AS VARCHAR(16))
		PRINT @MSG
		END
	ELSE
		BEGIN	
		-- Get the column of the old boy in her preference list 
		SET @SQL = N'
		SELECT @OUT = 
		(
		SELECT X
		FROM G
		UNPIVOT ( B FOR X IN ( ' + @ColumnNamesB + ' ) ) AS u
		WHERE G = @G AND B = @B)' 
		SET @ParmDefinition = N'@G int, @B int, @OUT NVARCHAR(16) OUTPUT'
		SET @INPUT1 = cast(@G AS NVARCHAR(16))
		SET @INPUT2 = cast(@B1 AS NVARCHAR(16))
		EXEC dbo.sp_executesql @SQL, @ParmDefinition, @G = @INPUT1, @B = @INPUT2, @OUT = @COL1 OUTPUT;

		-- Get the column of the new boy in her preference list
		SET @SQL = N'
		SELECT @OUT = 
		(
		SELECT X
		FROM G
		UNPIVOT ( B FOR X IN ( ' + @ColumnNamesB + ' ) ) AS u
		WHERE G = @G AND B = @B)' 
		SET @ParmDefinition = N'@G int, @B int, @OUT NVARCHAR(16) OUTPUT'
		SET @INPUT1 = cast(@G AS NVARCHAR(16))
		SET @INPUT2 = cast(@B AS NVARCHAR(16))
		EXEC dbo.sp_executesql @SQL, @ParmDefinition, @G = @INPUT1, @B = @INPUT2, @OUT = @COL2 OUTPUT;

		-- Drop the old boy for the new boy if he appears earlier on her preference list 
		-- Otherwise, do nothing
		IF @COL2 < @COL1
			BEGIN
				-- Drop the old boy
				SET @SQL = N'DELETE FROM BG WHERE B = ' + cast(@B1 AS NVARCHAR(16)) 
				EXEC dbo.sp_executesql @SQL

				-- Comment
				SET @MSG =	'Fortunately for him, Girl ' + cast(@G AS VARCHAR(16)) + 
							' prefers Boy ' + cast(@B AS VARCHAR(16)) + ' to current Boy ' + 
							cast(@B1 AS VARCHAR(16))
				PRINT @MSG

				-- Become engaged to the new boy
				INSERT INTO BG(B,G) VALUES (@B,@G)

				-- Comment
				SET @MSG =	'So Girl ' + cast(@G AS VARCHAR(16)) + ' drops Boy ' + 
							cast(@B1 AS VARCHAR(16)) + 	' and becomes engaged to Boy ' + 
							cast(@B AS VARCHAR(16))
				PRINT @MSG
			END
		ELSE
			BEGIN	
			-- Comment	
			SET @MSG =	'Unfortunately, Girl ' + cast(@G AS VARCHAR(16)) + ' prefers Boy ' + 
						cast(@B1 AS VARCHAR(16)) + ' to Boy ' + cast(@B AS VARCHAR(16))
			PRINT @MSG
			END
		END

	-- NULLIFY the column of the girl to whom the boy just proposed, so he won't do it again
	SET @SQL = N'UPDATE B SET ' + @COL + ' = NULL WHERE B = ' + cast(@B AS NVARCHAR(16))
	exec sp_executesql @SQL

	END

-- Debug, if necessary
IF @DEBUG = 1 
	BEGIN
	SELECT * FROM B
	END

-- Display remaining answer
SELECT * FROM G
SELECT * FROM BG ORDER BY B

END

GO


